' Turing.bas - A Turing Machine Simulator
' Rev 1.0.0 William M Leue 12/7/2020

option default integer
option base 1

' Fundamental simulator constants
const NUM_ADDRESSES = 16
const NUM_TAPE_CELLS = 10000
const TAPE_START = NUM_TAPE_CELLS\2
const NUM_SYMBOLS = 4
const NUM_MICRO_INSTR = 3
const NUM_INSTR_PARTS = NUM_SYMBOLS*NUM_MICRO_INSTR
const NORMAL_PART_LENGTH = 2*NUM_MICRO_INSTR
const READY = 1
const HALTED = 2
const RUNNING = 3
const PAUSED = 4
const STEPPING = 5
const NSTATES = 5
const HALT = -1
const NOT_USED = -2
const L = 0
const R = 1
const MAX_DELAY = 5000
const PDIR = "./Programs/"
const ERR_NONE = 0
const ERR_TAPEL = 1
const ERR_TAPER = 2
const NUM_ERRS = 2

' Menu screen constants
const PLOAD = 1
const PSAVE = 2
const PENTER = 3
const TABULAR = 4
const GRAPHICAL = 5
const HELP = 6
const OPTIONS = 7
const QUIT = 8
const NMENUS = 8
const MBHEIGHT = 40
const MBWIDTH = 400
const MBYOFF = 100

' Commands used in various places
const DELETE = 127
const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const ENTER = 13
const XMENU = 77     'M'
const XMENUL = 109   'm'
const RCLEAR = 67    'C'
const RCLEARL = 99   'c'
const RNAME = 78     'N'
const RNAMEL = 110   'n'
const RSTART = 83    'S'
const RSTARTL = 115  's'
const RTAPEM = 84    'T'
const RTAPEML = 116  't'
const ESCAPE = 27
const XFIELD = 88    'X'
const XFIELDL = 120  'x'

' Tape Display Constants
const TYOFF = 20
const HWIDTH = 65
const HHEIGHT = 40
const HYOFF = 10
const SMALL = 1
const MEDIUM = 2
const LARGE = 3

' Tabular Screen Constants
const TBXOFF = 20
const TBYOFF = 120
const ROWH = 20
const ACOLW = 40
const WMCOLW = 20
const BAN1H = 30
const BAN2H = 30
const NPSECS = 4
const PSW = 2*WMCOLW+ACOLW
const TSECW = NPSECS*PSW
const TBTW = ACOLW+TSECW
const TBTH = BAN1H+BAN2H+NUM_ADDRESSES*ROWH

' Tabular Commands Constants
const CMDSX = 420
const CMDSY = TBYOFF
const CMDSW = 360
const CMDSH = TBTH+30
const CMDPX = 200
const RUNSTOP = 1
const RSTEP = 2
const RESET = 3
const RMENU = 4
const NCMDS = 4
const CMX = CMDSX+20
const CMY = CMDSY+210
const CMH = 40
const CMW = 280

' Graphical Screen Constants
const GRYTOP = 120
const GRYBOT = 550
const GRCRAD = 150
const GRCASP = 1.8
const GRNRAD = 30
const GRSARAD = 30
const NATYPE = 1
const SATYPE = 2
const NATYPES = 2
const ACOFFR = 0.15
const AHANG = 135.0
const AHLEN = GRNRAD+20
const ASBANG = 165.0
const ASFANG = 255.0
const NUM_KNOTS = 7
const SPLINE_SCALE = 1.0/6.0
const NUM_INTERP = 3
const ARC_CURVE_OFFSET = 300.0
const NUM_CURVE = 2*(NUM_KNOTS+3)*NUM_INTERP + 7
const ARROW_ANGLE = 18
const ARROW_ANGLE2 = 4
const ARROW_LENGTH = 10
const SELF_ARC_ANGLE_SPREAD = 120
const SELF_ARC_ANGLE_CORR = 4
const MAX_ARC_GROUPS = 64
const MIN_LABEL_OFFSET = -100
const MAX_LABEL_OFFSET = 100
const NTYPE = 1
const STYPE = 2
const HTYPE = 3

' Graphical Screen Commands
const MRUN = 1
const MSELECT = 2
const MADJUST = 3
const GRUN = 82
const GRUNL = 114
const GSTEP = 83
const GSTEPL = 115
const GRESET = 69
const GRESETL = 101
const GADJUST = 65
const GADJUSTL = 97
const GMENU = 77
const GMENUL = 109
const GNODE = 78
const GNODEL = 110
const GARC = 65
const GARCL = 97
const GLABLE = 76
const GLABLEL = 108

' Options Constants
const DELAY_CMD = 68
const TAPE_CMD = 84
const MENU_CMD = 77
const OMXOFF = 20
const OMYOFF = 100
const OMWIDTH = 600
const OMHEIGHT = 60
const NOMENUS = 3

' Enter a Turing Program by Hand Constants
const EPXOFF = 20
const EPYOFF = 50
const EDIT_PROGRAM = 1
const EDIT_TAPE = 2

' Help System
const NHELP_PAGES = 13

' Globals
dim mchoice = 0
dim prev_mchoice = 0
dim tape(NUM_TAPE_CELLS)
dim program(NUM_ADDRESSES, NUM_INSTR_PARTS)
dim start_address = 0
dim startTapeCell = NUM_TAPE_CELLS\2
dim current_address = start_address
dim current_tape_cell = startTapeCell
dim most_recent_tape_cell = 0
dim current_tape_value = 0
dim lowest_tape_cell_used = startTapeCell
dim highest_tape_cell_used = startTapeCell
dim max_address_used = start_address
dim moves$(2) = ("L", "R")
dim rmchoice = 0
dim prev_rmchoice = 0
dim state = HALTED
dim stateNames$(NSTATES) = ("READY", "HALTED", "RUNNING", "PAUSED", "STEPPING")
dim num_executions = 0
dim delay = 0
dim program_loaded = 0
dim programName$ = ""
dim programDesc$ = ""
dim tape_size = LARGE
dim tapeSizes$(3) = ("Small", "Medium", "Large")
dim twidth_sizes(3) = (20, 30, 40)
dim tcwidth_sizes(3) = (20, 30, 40)
dim tfont_sizes(3) = (7, 1, 5)
dim float tfont_offsets(3) = (0.9, 0.8, 0.7)
dim opchoice = 0
dim prev_opchoice = 0
dim eaddr_choice = 0
dim prev_eaddr_choice = 0
dim efield_choice = 0
dim prev_efield_choce = 0
dim program_path$ = ""
dim error_code = ERR_NONE
dim error_strings$(NUM_ERRS) = ("Off Tape to Left", "Off Tape to Right")
dim knots(NUM_KNOTS+3, 2)
dim points(NUM_CURVE, 2)
dim nlocs(NUM_ADDRESSES, 2)
dim node_selected = 0
dim arc_selected = 0
dim label_selected = 0
dim num_arc_groups = 0
dim arc_groups(MAX_ARC_GROUPS, 3)
dim arc_group_labels$(MAX_ARC_GROUPS)
dim float arc_angle_corr(MAX_ARC_GROUPS)
dim label_offsets(MAX_ARC_GROUPS, 2)
dim float bratio(MAX_ARC_GROUPS) 
dim uses_graphics = 0
dim last_help_page = 1
dim pnames$(NHELP_PAGES)

' Main Program
Setup
InitHelp
do
  MenuScreen cmd
  select case cmd
    case PLOAD
      LoadProgram
    case PSAVE
      SaveProgram
    case PENTER
      HandleEnterProgramEvents
    case TABULAR
      HandleTabularMenuEvents
    case GRAPHICAL
      HandleGraphicalEvents
    case HELP
      HandleHelpEvents last_help_page
    case OPTIONS
      HandleOptionsEvents
    case QUIT
      cls
      close #1
      end
  end select
loop
end

' Various initializations
sub Setup
  open "debug.txt" for output as #1
end sub

' Main Menu Screen
sub MenuScreen cmd
  local i, x, y, tx, ty
  cls
  text MM.HRES\2, 20, "Turing Machine Simulator", "CT", 5
  x = MM.HRES\2 - MBWIDTH\2
  for i = 1 to NMENUS
    y = MBYOFF + (i-1)*MBHEIGHT
    box x, y, MBWIDTH, MBHEIGHT
    tx = x+20
    ty = y+15
    select case i
      case PLOAD
        text tx, ty, "Load a Turing Program"
      case PSAVE
        text tx, ty, "Save a Turing Program"
      case PENTER
        text tx, ty, "Enter or Edit a Turing Program by Hand"
      case TABULAR
        text tx, ty, "Show and Run Program in Tabular Form"
      case GRAPHICAL
        text tx, ty, "Show and Run Program in Graphical Form"
      case HELP
        text tx, ty, "Show Help Screens"
      case OPTIONS
        text tx, ty, "Show Options Screen"
      case QUIT
        text tx, ty, "Quit"
    end select
  next i  
  
  mchoice = PLOAD
  prev_mchoice = PLOAD
  HighliteMenu x-5, mchoice
  GetMenuEvents cmd
end sub

' Highlite the currently selected Menu Item
sub HighliteMenu x, which
  local y
  if prev_mchoice > 0 then
    y = MBYOFF + (prev_mchoice-1)*MBHEIGHT + MBHEIGHT\2
    triangle x, y, x-20, y-20, x-20, y+20, RGB(BLACK), RGB(BLACK)
  end if
  y = MBYOFF + (which-1)*MBHEIGHT + MBHEIGHT\2
  triangle x, y, x-20, y-20, x-20, y+20, RGB(RED), RGB(RED)
  prev_mchoice = which
end sub

' Handle User Menu selection events
sub GetMenuEvents cmd
  local z$
  local x
  x = MM.HRES\2 - MBWIDTH\2 - 5

  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case DOWN
        mchoice = mchoice+1
        if mchoice > NMENUS then
          mchoice = 1
        end if
        HighliteMenu x, mchoice        
      case UP
        mchoice = mchoice-1
        if mchoice < 1 then
          mchoice = NMENUS
        end if
        HighliteMenu x, mchoice        
      case ENTER
        cmd = mchoice
        exit do
    end select
  loop 
end sub

' Load a Turing machine program from SD card
sub LoadProgram
  local pname$, buf$, fld$, w$
  local i, j, ok
  w$ = INKEY$
  cls
  do
      ok = 1
    input "Enter the Program Filename ('.tur' extension will be added): ", pname$
    cmd = asc(LEFT$(pname$, 1))
    if cmd = ESCAPE then
      exit sub
    end if
    if len(pname$) > 0 then
      if instr(pname$, ".tur") = 0 then
        pname$ = pname$ + ".tur"
        program_path$ = PDIR + pname$
      end if
    else
      print "You need to enter a filename, or press Escape Key to return to menu"
      ok = 0
    end if
    if ok then
      on error skip 1
      open program_path$ for input as #2
      if MM.ERRNO > 0 then
        print "Sorry, file '" + program_path$ + "' was not found. Please try again"
        ok = 0
      end if
    end if
  loop until ok = 1
  line input #2, buf$
  programName$ = buf$
  line input #2, buf$
  programDesc$ = buf$
  line input #2, buf$
  max_address_used = val(buf$)-1
  for i = 0 to max_address_used
    line input #2, buf$
    for j = 1 to NUM_INSTR_PARTS
      fld$ = FIELD$(buf$, j, ",")
      program(i+1, j) = val(fld$)
    next j
  next i
  line input #2, buf$
  current_tape_cell = val(buf$)
  line input #2, buf$  
  DecodeTape buf$
  line input #2, buf$
  delay = val(buf$)
  line input #2, buf$
  tape_size = val(buf$)
  line input #2, buf$
  uses_graphics = val(buf$)
  if uses_graphics then
    for i = 1 to max_address_used+1
      line input #2, buf$
      for j = 1 to 2
        fld$ = FIELD$(buf$, j, ",")
        nlocs(i, j) = val(fld$)
      next j
    next i
    line input #2, buf$
    num_arc_groups = val(buf$)
    for i = 1 to num_arc_groups
      line input #2, buf$
      bratio(i) = val(buf$)
    next i
    for i = 1 to num_arc_groups
      line input #2, buf$
      for j = 1 to 2
        fld$ = FIELD$(buf$, j, ",")
        label_offsets(i, j) = val(fld$)
      next j
    next i
    for i = 1 to num_arc_groups
      line input #2, buf$
      arc_angle_corr(i) = val(buf$)
    next i
  end if
  close #2 
  program_loaded = 1
  state = READY
end sub

' Reload Program for Reset
' This only reloads the tape and program address,
' no need to reload the graphics stuff.
sub ReloadProgram
  open program_path$ for input as #2
  line input #2, buf$
  programName$ = buf$
  line input #2, buf$
  programDesc$ = buf$
  line input #2, buf$
  max_address_used = val(buf$)-1
  for i = 0 to max_address_used
    line input #2, buf$
    for j = 1 to NUM_INSTR_PARTS
      fld$ = FIELD$(buf$, j, ",")
      program(i+1, j) = val(fld$)
    next j
  next i
  line input #2, buf$
  current_tape_cell = val(buf$)
  line input #2, buf$  
  DecodeTape buf$
  line input #2, buf$
  delay = val(buf$)
  line input #2, buf$
  tape_size = val(buf$)
  close #2 
end sub

' Save a Turing machine program to SD card
sub SaveProgram
  local pname$, w$, prompt$
  local i, j, ok
  cls
  w$ = INKEY$
  do
    ok = 1
    input "Enter the Program Filename ('.tur' extension will be added): ", pname$
    if len(pname$) > 0 then
      if asc(LEFT$(pname$,1)) = ESCAPE then
        exit sub
      end if
      if instr(pname$, ".tur") = 0 then
        pname$ = pname$ + ".tur"
        program_path$ = PDIR$ + pname$
        on error skip 1
        open program_path$ for input as #2
        if MM.ERRNO = 0 then
          close #2
          print "File '" + program_path$ + "' already exists. Overwrite? (Y,N): ";
          input w$
          if LEFT$(UCASE$(w$),1) = "N" then
            ok = 0
          end if
        end if
      end if
    else
      print "You need to enter a filename or press Escape Key to return to menu"
      ok = 0
    end if
    if ok then
      on error skip 1
      open program_path$ for output as #2
      if MM.ERRNO > 0 then
        print "Could not open '" + program_path$ + "' for writing, try again"
        ok = 0
      end if
    end if
  loop until ok = 1
  print #2, programName$
  print #2, programDesc$
  print #2, max_address_used+1
  for i = 0 to max_address_used
    for j = 1 to NUM_INSTR_PARTS
      print #2, str$(program(i+1, j)) + ",";
    next j
    print #2, ""
  next i
  print #2, str$(current_tape_cell)
  print #2, EncodeTape$()
  print #2, str$(delay)
  print #2, str$(tape_size)
  print #2, str$(uses_graphics)
  if uses_graphics then
    for i = 1 to max_address_used+1
      print #2, str$(nlocs(i, 1)) + "," + str$(nlocs(i, 2)) + ","
    next i
    print #2, str$(num_arc_groups)
    for i = 1 to num_arc_groups
      print #2, str$(bratio(i))
    next i
    for i = 1 to num_arc_groups
      print #2, str$(label_offsets(i, 1)) + "," + str$(label_offsets(i, 2)) + ","
    next i
    for i = 1 to num_arc_groups
      print #2, str$(arc_angle_corr(i))
    next i
  end if
  close #2
end sub

' Encode Tape Values in Run-length format
function EncodeTape$()
  local i, v, rlen, f
  local out$ = ""
  v = tape(1)
  rlen = 1
  f = 1
  for i = 2 to NUM_TAPE_CELLS
    if tape(i) = v then
      rlen = rlen+1
    else
      out$ = out$ + str$(rlen) + ":" + str$(v) + ","
      v = tape(i)
      rlen = 1
      f = i
    end if
  next i
  out$ = out$ + str$(rlen) + ":" + str$(v) + ","
  EncodeTape$ = out$
end function

' Decode Run-Length-Encoded Tape values onto Tape
sub DecodeTape rle$
  local i, j, p, rlen, v
  local spec$
  ClearTape
  p = 1
  i = 1
  do
    spec$ = FIELD$(rle$, i, ",") + ":"
    if len(spec$) > 2 then
      i = i+1
      rlen = val(FIELD$(spec$, 1, ":"))
      v = val(FIELD$(spec$, 2, ":"))
      for j = 1 to rlen
        tape(p+j-1) = v
      next j
      p = p+rlen
    end if
  loop until len(spec$) < 2
end sub

' Test the Tape RLE Encoding and Decoding (debug only)
sub TestTapeEnDecode
  local i
  local code$
  ClearTape
  code$ = EncodeTape$()
  print "Encoding for a blank tape:  " + code$
  for i = 5000 to 5009
    tape(i) = 1
  next i
  tape(5010) = 0
  for i = 5011 to 5016
    tape(i) = 1
  next i
  code$ = EncodeTape$()
  print "Encoding for a tape with two fields of 1:  " + code$
  DecodeTape code$
  code$ = EncodeTape$()
  print "Re-encoding after decoding: " + code$
end sub

' Draw a Turing machine program in Tabular form
' and Allow running and Stepping it
sub DrawTabularScreen x
  page write 1
  cls
  DrawTape current_tape_cell
  current_tape_value = tape(current_tape_cell)
  DrawTable
  FillTable
  DrawTabularCommands x
  page write 0
  page copy 1 to 0, B
end sub

' Handle user inputs for Graphical screen
sub HandleGraphicalEvents
  local z$
  local cmd
  local mmode = MRUN
  local pstate = 0
  
  if not uses_graphics then
    PlaceNodes
  end if
  uses_graphics = 1
  state = READY
  current_tape_value = tape(current_tape_cell)
  MakeArcGroups  
  DrawGraphicalScreen mmode
  do
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(Z$))
    select case mmode
      case MRUN
        select case cmd
          case GRUN, GRUNL
            if state = READY then
              PrepForRun
            end if
            if state = READY or state = PAUSED then
              state = RUNNING
              z$ = INKEY$
              do
                z$ = INKEY$
                if z$ <> "" then
                  state = PAUSED
                  DrawGraphicalScreen mmode, "(R)un"
                  exit do
                end if
                Cycle
                DrawGraphicalScreen mmode, "(P)ause"
                pause delay
              loop until state = HALTED
            end if  
          case GSTEP, GSTEPL
            pstate = state
            if state = READY or state = PAUSED then
              state = STEPPING
              Cycle
              DrawGraphicalScreen mmode
              pause 100
              if state = STEPPING then
                state = pstate
              end if
            end if
          case GRESET, GRESETL
            ReloadProgram
            PrepForRun
            DrawGraphicalScreen mmode
          case GADJUST, GADJUSTL
            mmode = MSELECT
            DrawGraphicalScreen mmode
          case GMENU, GMENUL
            state = HALTED
            exit do
        end select
      case MSELECT
        select case cmd
          case GNODE, GNODEL
            arc_selected = 0
            label_selected = 0
            node_selected = node_selected+1
            if node_selected > max_address_used+1 then
              node_selected = 1
            end if
            DrawGraphicalScreen mmode
          case GARC, GARCL
            node_selected = 0
            label_selected = 0
            arc_selected = arc_selected+1
            if arc_selected > num_arc_groups then
              arc_selected = 1
            end if
            DrawGraphicalScreen mmode
          case GLABLE, GLABLEL
            arc_selected = 0
            node_selected = 0
            label_selected = label_selected+1
            if label_selected > num_arc_groups then
              label_selected = 1
            end if
            DrawGraphicalScreen mmode
          case GMENU, GMENUL
            node_selected = 0
            arc_selected = 0
            label_selected = 0
            mmode = MRUN
            DrawGraphicalScreen mmode
          case ENTER
            mmode = MADJUST
            DrawGraphicalScreen mmode
        end select
      case MADJUST
        select case cmd
          case UP
            if node_selected > 0 then
              nlocs(node_selected, 2) = nlocs(node_selected,2) - 10
              if nlocs(node_selected, 2) < GRYTOP then
                nlocs(node_selected, 2) = GRYTOP
              end if
              DrawGraphicalScreen mmode
            end if           
            if arc_selected > 0 then
              if arc_groups(arc_selected, 3) = NTYPE then
                bratio(arc_selected) = bratio(arc_selected) + 0.1
                if bratio(arc_selected) > 2.0 then 
                  bratio(arc_selected) = 2.0
                end if
              else
                arc_angle_corr(arc_selected) = arc_angle_corr(arc_selected)+10
              end if
              DrawGraphicalScreen mmode
            end if
            if label_selected > 0 then
              label_offsets(label_selected, 2) = label_offsets(label_selected, 2) - 1
              if label_offsets(label_selected, 2) < MIN_LABEL_OFFSET then
                label_offsets(label_selected, 2) = MIN_LABEL_OFFSET
              end if              
              DrawGraphicalScreen mmode
            end if
          case DOWN
            if node_selected > 0 then
              nlocs(node_selected, 2) = nlocs(node_selected,2) + 10
              if nlocs(node_selected, 2) > GRYBOT then
                nlocs(node_selected, 2) = GRYBOT
              end if
              DrawGraphicalScreen mmode
            end if
            if arc_selected > 0 then
              if arc_groups(arc_selected, 3) = NTYPE then
                bratio(arc_selected) = bratio(arc_selected) - 0.1
                if bratio(arc_selected) < -2.0 then bratio(arc_selected) = -2.0
              else
                arc_angle_corr(arc_selected) = arc_angle_corr(arc_selected)-10
              end if
              DrawGraphicalScreen mmode
            end if
            if label_selected > 0 then
              label_offsets(label_selected, 2) = label_offsets(label_selected, 2) + 1
              if label_offsets(label_selected, 2) > MAX_LABEL_OFFSET then
                label_offsets(label_selected, 2) = MAX_LABEL_OFFSET
              end if              
              DrawGraphicalScreen mmode
            end if
          case LEFT
            if node_selected > 0 then
              nlocs(node_selected, 1) = nlocs(node_selected, 1) - 10
              if nlocs(node_selected, 1) < GRNRAD+10 then
                nlocs(node_selected, 1) = GRNRAD+10
              end if
              DrawGraphicalScreen mmode
            end if  
            if arc_selected > 0 then
              if bratio(arc_selected) >= 0.0 then bratio(arc_selected) = -bratio(arc_selected)
              DrawGraphicalScreen mmode
            end if
            if label_selected > 0 then
              label_offsets(label_selected, 1) = label_offsets(label_selected, 1) - 1
              if label_offsets(label_selected, 1) < MIN_LABEL_OFFSET then
                label_offsets(label_selected, 1) = MIN_LABEL_OFFSET
              end if              
              DrawGraphicalScreen mmode
            end if
          case RIGHT
            if node_selected > 0 then
              nlocs(node_selected, 1) = nlocs(node_selected, 1) + 10
              if nlocs(node_selected, 1) > MM.HRES-GRNRAD-10 then
                nlocs(node_selected, 1) = MM.HRES-GRNRAD-10
              end if
              DrawGraphicalScreen mmode
            end if  
            if arc_selected > 0 then
              if bratio(arc_selected) < 0.0 then bratio(arc_selected) = -bratio(arc_selected)
              DrawGraphicalScreen mmode
            end if
            if label_selected > 0 then
              label_offsets(label_selected, 1) = label_offsets(label_selected, 1) + 1
              if label_offsets(label_selected, 1) > MAX_LABEL_OFFSET then
                label_offsets(label_selected, 1) = MAX_LABEL_OFFSET
              end if              
              DrawGraphicalScreen mmode
            end if
          case ENTER
            mmode = MSELECT
            DrawGraphicalScreen mmode
        end select
    end select

  loop
end sub

' Do initial Node placement
sub PlaceNodes
  local i, xc, yc
  local float cang = 180.0

  xc = MM.HRES\2
  yc = (GRYTOP+GRYBOT)\2
  for i = 1 to max_address_used+1
    nlocs(i, 1) = xc + cos(rad(cang))*GRCRAD*GRCASP
    nlocs(i, 2) = yc - sin(rad(cang))*GRCRAD
    cang = cang - 360.0/(max_address_used+1.0)
  next i
end sub

' Group Arcs so that multiple arcs from one node to another
' are comsolidated into one arc group with multiple tape
' read symbol triggers.
' Each arc group has 3 components: Source Node, Destination Node,
' and Arc Type. (For Self and Halt arcs, source and destination
' are the same.) Types: Normal, Self, and Halt
sub MakeArcGroups
  local addr, symbol, p, nxtaddr, g, hit
  num_arc_groups = 0
  for addr = 0 to max_address_used
    for symbol = 1 to NUM_SYMBOLS
      p = 1 + (symbol-1)*NUM_MICRO_INSTR
      nxtaddr = program(addr+1, p+2)
      if nxtaddr <> NOT_USED then
        if nxtaddr = HALT then
            hit = 0
            num_arc_groups = num_arc_groups+1
            arc_groups(num_arc_groups, 1) = addr
            arc_groups(num_arc_groups, 2) = nxtaddr
            arc_groups(num_arc_groups, 3) = HTYPE
            arc_group_labels$(num_arc_groups) = str$(symbol-1)
        else
          hit = 0
          for g = 1 to num_arc_groups
            if arc_groups(g, 1) = addr and arc_groups(g, 2) = nxtaddr then
              hit = 1
              arc_group_labels$(g) = arc_group_labels$(g) + str$(symbol-1)
              exit for
            end if
          next g
          if not hit then
            num_arc_groups = num_arc_groups+1
            arc_groups(num_arc_groups, 1) = addr
            arc_groups(num_arc_groups, 2) = nxtaddr
            if nxtaddr = addr then
              arc_groups(num_arc_groups, 3) = STYPE
            else
              arc_groups(num_arc_groups, 3) = NTYPE
            end if
            arc_group_labels$(num_arc_groups) = str$(symbol-1)
          end if
        end if
      end if
    next symbol
  next addr
  for i = 1 to num_arc_groups
    label$ = arc_group_labels$(i)
    nlabel$ = ""
    jlast = len(label$)
    for j = 1 to jlast
      nlabel$ = nlabel$ + MID$(label$, j, 1)
      if j < jlast then
        nlabel$ = nlabel$ + ","
      end if
    next j
    arc_group_labels$(i) = nlabel$
  next i
end sub

' Draw a Turing machine program in Graphical form
' and allow running and stepping it
sub DrawGraphicalScreen mmode, label$
  local xc, yc, x, y, i, j, k, src, dst, daddr
  local xs, ys, c, tval

  page write 1
  cls
  box 0, GRYTOP, MM.HRES-1, GRYBOT-GRYTOP-1
  DrawTape current_tape_cell
  tval = current_tape_value
  xc = MM.HRES\2
  yc = (GRYTOP+GRYBOT)\2

  ' Draw the Arcs
  for i = 1 to num_arc_groups
    src = arc_groups(i, 1)
    dst = arc_groups(i, 2)
    c = RGB(WHITE)
    if current_address = src and IsArcActivated(i, tval) then
      c = RGB(RED)
    end if
    alabel$ = arc_group_labels$(i)
    if arc_selected = i then
      c = RGB(YELLOW)
    end if
    if dst = HALT then
      DrawHaltArc i, src, nlocs(), AHANG, c
    else if dst = src then
      DrawSelfArc i, src, nlocs(), alabel$, c
    else
      DrawNormalArc i, src, dst, nlocs(), alabel$, c
    end if
  next i

  ' Draw the Nodes
  for i = 1 to max_address_used+1
    c = RGB(WHITE)
    if current_address = i-1 then
      c = RGB(RED)
    end if
    DrawNode i, nlocs(), c
  next i

  ' Draw the Commands
  DrawGraphicalCommands mmode, label$

  page write 0
  page copy 1 to 0, B
end sub

' Draw an Arc between 2 nodes
' Arcs are drawn as a smooth curve using knot locations
' calculated as offsets from the straight line between
' the nodes and BSpline interpolation between the knots.
sub DrawNormalArc which, src, dst, nlocs(), alabel$, color
  local knots(NUM_KNOTS+3, 2)
  local points(NUM_CURVE, 2)
  local ox, oy, nx, ny, i, np, ls, ly
  local float angle, acorr, dx, dy, lc
    
  MakeKnots which, src, dst, nlocs(), knots()
  BSPlineCurve knots(), points(), np
  ox = points(1, 1) : oy = points(1, 2)
  for i = 2 to np
    nx = points(i, 1) : ny = points(i, 2)
    line ox, oy, nx, ny,, color
    ox = nx : oy = ny
  next i
  GetDestinationPosition points(), np, dx, dy, angle
  DrawArrowhead dx, dy, angle, color
  GetLabelPosition which, points(), np, lx, ly
  lc = color
  if label_selected = which then
    lc = RGB(YELLOW)
  end if
  text lx, ly, alabel$,,,, lc
end sub

' Draw an Arc from a Node back to the same Node
' src is the index of the node.
' cangle is the angle of the arc around the node center.
sub DrawSelfArc which, src, nlocs(), alabel$, color
  local xs, ys, ls, ly, lxoff, lyoff
  local float acx, acy, arx1, ary1, arx2, ary2, ara1, a1, a2
  local float arad = GRSARAD
  local float tlen = ARROW_LENGTH

  ' node coordinates
  xs = nlocs(src+1, 1) : ys = nlocs(src+1, 2)

  ' point on circumference of node circle with current angle
  acorr = arc_angle_corr(which)
  acx = xs + cos(rad(acorr))*GRNRAD
  acy = ys - sin(rad(acorr))*GRNRAD

  ' approximate point of intersection of arc with node circle
  a1 = acorr - SELF_ARC_ANGLE_SPREAD
  arx1 = acx + cos(rad(a1))*GRSARAD
  ary1 = acy - sin(rad(a1))*GRSARAD
  a2 = acorr + SELF_ARC_ANGLE_SPREAD
  arx2 = acx + cos(rad(a2))*GRSARAD
  ary2 = acy - sin(rad(a2))*GRSARAD

  ' arc
  circle acx, acy, GRSARAD,,, color

  ' arrowhead
  DrawSelfArcArrowhead arx1, ary1, a1+90+SELF_ARC_ANGLE_CORR, color

  ' label
  GetSelfLabelPosition which, xs, ys, + a1-180, lx, ly
  lc = color
  if label_selected = which then
    lc = RGB(YELLOW)
  end if
  lxoff = label_offsets(which, 1) : lyoff = label_offsets(which, 2)
  text lx+lxoff, ly+lyoff, alabel$, "CM",,, lc
end sub

' Draw the 'Halt' Arc from a Node
sub DrawHaltArc which, src, nlocs(), cangle, color
  local xs, ys, x, y
  local float acorr

  xs = nlocs(src+1, 1) : ys = nlocs(src+1, 2)
  acorr = arc_angle_corr(which)
  x = xs + cos(rad(cangle+acorr))*AHLEN
  y = ys - sin(rad(cangle+acorr))*AHLEN
  line xs, ys, x, y,, color
  DrawHaltArrowhead x, y, cangle+acorr+180.0, color
  text x, y, "H", "RB",,, color
end sub  

' Draw an Arrowhead for a Node-to-Node Arc
sub DrawArrowhead x, y, angle as float, color
  local tlen
  local a1, a2
  local brx1, bry1, brx2, bry2, brx3, bry3

  brx1 = x + cos(rad(angle))*GRNRAD
  bry1 = y - sin(rad(angle))*GRNRAD

  a1 = angle - ARROW_ANGLE
  a2 = angle + ARROW_ANGLE
  tlen = ARROW_LENGTH
  
  brx2 = brx1 + cos(rad(a1))*(tlen)
  bry2 = bry1 - sin(rad(a1))*(tlen)
  brx3 = brx1 + cos(rad(a2))*(tlen)
  bry3 = bry1 - sin(rad(a2))*(tlen)
  triangle brx1, bry1, brx2, bry2, brx3, bry3, color, color
end sub

' Draw an Arrowhead for the Halt Arc
sub DrawHaltArrowhead x, y, angle as float, color
  local tlen
  local a1, a2
  local brx1, bry1, brx2, bry2, brx3, bry3

  a1 = angle - ARROW_ANGLE2*3
  a2 = angle + ARROW_ANGLE2*3
  tlen = ARROW_LENGTH
  
  brx2 = x + cos(rad(a1))*tlen
  bry2 = y - sin(rad(a1))*tlen
  brx3 = x + cos(rad(a2))*tlen
  bry3 = y - sin(rad(a2))*tlen
  triangle x, y, brx2, bry2, brx3, bry3, color, color
end sub

' Draw an Arrowhead for a Self Arc
sub DrawSelfArcArrowhead x, y, angle as float, color
  local tlen
  local a1, a2
  local brx1, bry1, brx2, bry2, brx3, bry3

  a1 = angle - ARROW_ANGLE2*3
  a2 = angle + ARROW_ANGLE2*3
  tlen = ARROW_LENGTH
  
  brx2 = x + cos(rad(a1))*tlen
  bry2 = y - sin(rad(a1))*tlen
  brx3 = x + cos(rad(a2))*tlen
  bry3 = y - sin(rad(a2))*tlen
  triangle x, y, brx2, bry2, brx3, bry3, color, color
end sub

' Get coords to put source symbol label for a normal arc
sub GetLabelPosition which, points(), np, lx, ly
  local cx, cy, px, py
  local i, dx, dy, q, dist
  local float angle
  cx = points(1, 1) : cy = points(1, 2)
  for i = 5 to np
    px = points(i, 1) : py = points(i, 2)    
    dx = px - cx : dy = py - cy
    dist = sqr(dx*dx + dy*dy)
    if dist >= GRNRAD+7 then
      exit for
    end if
  next i
  if dy < 0 then dy = -dy
  angle = deg(atan2(dy, dx))
  if dy > cy then
    angle = 360.0 - angle
  end if
  q = 0
  if angle >= 0 and angle <= 90 then q = 1
  if angle > 90 and angle <= 180 then q = 2
  if angle > 180 and angle <= 270 then q = 3
  if angle > 270 and angle <= 360 then q = 4
  lxo = label_offsets(which, 1) : lyo = label_offsets(which, 2)
  select case q
    case 1
      lx = px + 10 + lxo : ly = py - 20 + lyo
    case 2
      lx = px - 20 + lxo : ly = py - 20 + lyo
    case 3
      lx = px - 10 + lxo : ly = py + 20 + lyo
    case 4
      lx = px + 15 + lxo : ly = py - 10 + lyo
  end select  
end sub

' Get coords to put source symbol label for a self arc
sub GetSelfLabelPosition which, xc, yc, angle, lx, ly
  local lrad = GRNRAD+5
  lx = xc + cos(rad(angle))*lrad
  ly = yc - sin(rad(angle))*lrad
end sub
  
' Get the approximate destination point and angle
sub GetDestinationPosition points(), np, xp as float, yp as float, angle as float
  local px, ix, iy, dist, mdist, ps
  local float xlast, ylast, tangle, dx, dy, xr, yr

  ' center point of node
  xp = points(np-2, 1) : yp = points(np-2, 2)
  
  ' find point on entry curve closest to node circumference
  px = np-4
  xt = points(px, 1) : yt = points(px, 2)
  mdist = MM.HRES
  for i = px to 3 step -1
    xt = points(i, 1) : yt = points(i, 2)
    dist = int(sqr((xt-xp)*(xt-xp) + (yt-yp)*(yt-yp)))
    if dist > GRNRAD and dist < mdist then
      mdist = dist
      ps = i
    end if
  next i
  xt = points(ps, 1) : yt = points(ps, 2)
    
  ' back away by one point index
  xr = points(ps-1, 1) : yr = points(ps-1, 2)

  ' compute approx entry angle of arc to node
  dx = xr - xp : dy = yr - yp
  if dy < 0 then dy = -dy
  tangle = deg(atan2(dy, dx))
  if yr > yt then tangle = 360.0 - tangle
  angle = tangle

end sub

' Returns 1 if the specifed arc group is activated with
' the specified tape symbol read.
function IsArcActivated(which, tval)
  local alabel$
  local avals(max_address_used+1)
  local i, p, active

  active = 0
  alabel$ = arc_group_labels$(which)
  if instr(alabel$, ",") > 0 then
    alabel$ = alabel$ + ","
    p = 0
    do
      p = p+1
      temp$ = FIELD$(alabel$, p, ",")
      avals(p) = val(temp$)
    loop until len(temp$) = 0
    for i = 1 to p-1
      if avals(i) = tval then
        active = 1
        exit for
      end if
    next i
  else
    avals(1) = val(alabel$)
    if avals(1) = tval then
      active = 1
    end if
  end if
  IsArcActivated = active
end function

' Draw a Node
sub DrawNode which, nlocs(), color
  local xc, yc, c
  c = color
  if node_selected = which then
    c = RGB(YELLOW)
  end if
  xc = nlocs(which, 1) : yc = nlocs(which, 2)
print #1, "DrawNode which: " + str$(which) " xc: " + str$(xc) + " yc: " + str$(yc)

  circle xc, yc, GRNRAD,,, c, RGB(BLACK)
  text xc, yc, str$(which-1), "CM", 5,, c
end sub

' src and dst are the indices to the nodes connected by an arc
' The knots() array holds the coordinates of the created knots.
sub MakeKnots which, src, dst, nodes(), knots()
  local float dx, dy, jlen, jangle
  local k, xstart, ystart, xend, yend, offset
  local poff, xoff, yoff, linearx, lineary, curvedx, curvedy
  local float lratio, lt, ct

  ' get the curve endpoint coordinates
  xstart = nodes(src+1, 1) : ystart = nodes(src+1, 2)
  xend = nodes(dst+1, 1) : yend = nodes(dst+1, 2)
  
  ' treble the knots at each end
  knots(1, 1) = xstart : knots(1, 2) = ystart
  knots(2, 1) = xstart : knots(2, 2) = ystart
  knots(3, 1) = xstart : knots(3, 2) = ystart
  knots(NUM_KNOTS+1, 1) = xend : knots(NUM_KNOTS+1, 2) = yend
  knots(NUM_KNOTS+2, 1) = xend : knots(NUM_KNOTS+2, 2) = yend
  knots(NUM_KNOTS+3, 1) = xend : knots(NUM_KNOTS+3, 2) = yend

  ' compute the angle and distance between nodes
  dx = xend - xstart : dy = yend - ystart
  if dy < 0 then dy = -dy
  jlen = sqr(dx*dx + dy*dy)
  jangle = deg(atan2(dy, dx)) + 90.0
  if dy > ystart then jangle = 360.0 - jangle

  ' compute the offset for the bend
  lratio = jlen/400.0

  ' fill in the remaining knots using parametric vars lt and ct
  for k = 4 to NUM_KNOTS
    lt = (1.0*(k-4))/(NUM_KNOTS-4)
    ct = lt
    if ct > 0.5 then
      ct = (1.0*(NUM_KNOTS-k))/(NUM_KNOTS-4)
    end if
    offset = int(bratio(which)*ct*lratio*ARC_CURVE_OFFSET + 0.5)    
    xoff = int(offset*cos(rad(jangle)) + 0.5)
    yoff = -int(offset*sin(rad(jangle)) - 0.5)
    linearx = xstart + int(lt*(xend - xstart) + 0.5)
    lineary = ystart + int(lt*(yend - ystart) + 0.5)
    curvedx = linearx - xoff
    curvedy = lineary - yoff
    knots(k, 1) = curvedx : knots(k, 2) = curvedy
  next k
end sub

' Interpolate between the knots using BSpline interpolation
' to get a smooth curve.
sub BSplineCurve knots(), points(), np
  local n, x, y, k
  local float t, tstep, wm1, wm2, sm1, w0, wp1, tm2, tm1, tp1
  local float a = SPLINE_SCALE
  n = NUM_KNOTS + 2
  tstep = 1\(NUM_INTERP+1)
  if tstep = 0.0 then tstep = 0.1
  np = 1
  points(np, 1) = knots(1, 1) : points(np, 2) = knots(1, 2)
  x = points(np, 1) : y = points(np, 2)
  np = np+1
  for k = 3 to n-1
    for t = 0.0 to 1.0 step tstep
     wm2 = 0.0 : wm1 = 0.0 : wp1 = 0.0
      tp1 = t + 1.0
      tm1 = t - 1.0
      tm2 = t - 2.0
      wm2 = a * (2-tp1)*(2-tp1)*(2-tp1)
      wm1 = a * (4 - 6*t*t + 3*t*t*t)
      w0  = a * (4 - 6*tm1*tm1 - 3*tm1*tm1*tm1)
      wp1 = a * (2+tm2) * (2+tm2) * (2+tm2)
      x = int(wm2 * knots(k-2, 1) + wm1*knots(k-1, 1) + w0*knots(k, 1) + wp1*knots(k+1, 1))
      y = int(wm2 * knots(k-2, 2) + wm1*knots(k-1, 2) + w0*knots(k, 2) + wp1*knots(k+1, 2))
      points(np, 1) = x : points(np, 2) = y : np = np+1
    next t
  next k
  np = NUM_CURVE
end sub

' Draw the Graphical Commands
sub DrawGraphicalCommands which, label$
  local x, y
  x = 10
  y = GRYBOT+10
  rlabel$ = label$
  if len(label$) = 0 then
    rlabel$ = "(R)un"
  end if
  select case which
    case 1
      text x, y, rlabel$ + "   (S)tep   r(E)eset    (A)djust    (M)enu"
      text x+450, y, "Num Steps: " + str$(num_executions)
      text x+600, y, "Program State: " + stateNames$(state)
    case 2
      c$ = "(N): Adjust Node  (A): Adjust Arc  (L): Adjust Label  (M) Back to Run Menu"
      d$ = "Enter: Begin Adjusting Chosen Item Position
      text x, y, c$
      text x, y+15, d$
    case 3
      text x, y, "Use arrow keys to move selected item(s), press Enter when done"
  end select
end sub
  
' Draw the Tape and Read/Write Head
sub DrawTape tpos
  local x, y, i, n, f, nvis, m, c, bc, v
  local twidth, tcellw
  local float tfoff

  y = TYOFF
  twidth = twidth_sizes(tape_size)
  tcellw = tcwidth_sizes(tape_size)
  f = tfont_sizes(tape_size)
  tfoff = tfont_offsets(tape_size)
 
  line 0, y, MM.HRES-1, y
  line i, y+twidth, MM.HRES-1, y+twidth
  nvis = MM.HRES\tcellw - 1
  for i = 1 to nvis+1
    x = (i-1)*tcellw+tcellw\2
    line x, y, x, y+twidth
  next i
  x = MM.HRES\2
  y = TYOFF+twidth+HYOFF
  box x-HWIDTH\2, y, HWIDTH, HHEIGHT
  triangle x, TYOFF+twidth+1, x-10, y, x+10, y,, RGB(BLUE)
  text x-HWIDTH\2+4, y+8, "Read-Write",, 7
  text x-HWIDTH\2+20, y+20, "Head",, 7
  m = most_recent_tape_cell
  for i = 1 to nvis
    n = tpos-nvis\2+i-1
    if n < 1 then n = 1
    if n > NUM_TAPE_CELLS then n = NUM_TAPE_CELLS
    v = tape(n)
    x = (i-1)*tcellw + int(tfoff*tcellw)
    y = 7
    if tape_size = LARGE then
      text x, y, str$(n),,7
    end if
    c = RGB(WHITE)
    bc = RGB(BLACK)
    if v = 2 then
      c = RGB(BLACK)
      bc = RGB(GREEN)
    else if v = 3 then
      c = RGB(BLACK)
      bc = RGB(RED)
    end if   
    text x, TYOFF+8, str$(v),, f,, c, bc
  next i
  x = 20
  y = TYOFF+twidth+8
  x = 10
  text x, y, "Lowest Tape Addr: " + str$(lowest_tape_cell_used)
  x = 600
  text x, y, "Highest Tape Addr: " + str$(highest_tape_cell_used)
end sub

' Draw the Program Table
' This just draws the rows, columns, and labels.
' The 'FillTable' subroutine fills in program values.
sub DrawTable
  local x, y, i, j, row, ye, xm, fc, bc
  y = TBYOFF
  x = TBXOFF
  text x+TBTW\2, y-15, "Program", "CT"
  box x, y, TBTW, TBTH
  text x+3, y+8, "Read"
  box x, y, TBTW, BAN1H
  box x, y+BAN1H, TBTW, BAN2H
  text x+3, y+BAN1H+8, "Addr"
  line x+ACOLW, y, x+ACOLW, y+TBTH
  xm = x+ACOLW
  for i = 1 to NPSECS-1
    fc = RGB(WHITE) : bc = RGB(BLACK)
    if i = NPSECS-1 then 
      fc = RGB(BLACK)
      bc = RGB(GREEN)
    end if
    text xm+PSW\2-5, y+8, str$(i-1),,,, fc, bc
    xm = xm+PSW
    line xm, y, xm, y+TBTH
  next i
  fc = RGB(BLACK) : bc = RGB(RED)
  text xm+PSW\2-5, y+8, str$(3),,,, fc, bc
  y = y+BAN1H
  ye = y+TBTH-BAN1H
  xm = x + ACOLW
  for i = 1 to NPSECS
    text xm+3, y+8, "W  M"
    text xm+44, y+8, "Next"
    line xm+WMCOLW, y, xm+WMCOLW, ye,, RGB(60, 60, 60)
    line xm+2*WMCOLW, y, xm+2*WMCOLW, ye,, RGB(60, 60, 60)
    xm = xm+PSW
  next i
  for row = 1 to NUM_ADDRESSES-1
    y = TBYOFF + BAN1H + BAN2H + row*ROWH
    line x, y, x+TBTW, y
    text x+9, y-15, str$(row-1)
  next row
  text x+9, y-15+ROWH, str$(15)
  x = TBXOFF + TBTW\2
  y = TBYOFF + TBTH + 8
  text x, y, "Current Statement Executed Highlighted in Red", "CT"
end sub

' Fill in program values in the tabular display
sub FillTable
  local addr, p, pm, w, m, a, x, y, xm, c, rc, cmd
  local mv$, nxad$
  local word(NUM_INSTR_PARTS)

  x = TBXOFF
  rc = RGB(BLUE)
  for addr = 0 to max_address_used
    y = TBYOFF + BAN1H + BAN2H + addr*ROWH
    math slice program(), addr+1,, word()
    for p = 0 to NUM_SYMBOLS-1
      c = RGB(CYAN)
      if addr = current_address and p = current_tape_value then
        c = RGB(RED)
      end if
      pm = p*NUM_MICRO_INSTR+1
      w = word(pm)
      if w <> NOT_USED then
        m = word(pm+1)
print #1, "m: " + str$(m)
        mv$ = moves$(m+1)
        a = word(pm+2)
        nxad$ = str$(a)
        if a = HALT then nxad$ = "H"
        xm = x + p*PSW + ACOLW
        if p = 3 then  
          text xm+7, y+4, str$(w),,,, rc
        else
          text xm+7, y+4, str$(w),,,, c
        end if
        xm = xm+WMCOLW
        text xm+7, y+4, mv$,,,, c
        xm = xm+WMCOLW
        text xm+7, y+4, nxad$,,,, c
      end if
    next p
  next addr
end sub

' Draw the Status and Commands Section
sub DrawTabularCommands x
  local y, i, cmd, c, px
  
  x = CMDSX : y = CMDSY
  text x+CMDSW\2, y-15, "Status and Commands", "CT"
  box x, y, CMDSW, CMDSH
  c = RGB(CYAN)
  px = CMDSX + CMDPX
  box px, y, CMDSW-px, CMDY-y, RGB(BLACK), RGB(BLACK)
  text x+10, y+15, "Program Name:"           : text px, y+15, programName$,,,, c
  text x+10, y+30, "Start Address:"          : text px, y+30, str$(startAddress),,,, c
  text x+10, y+45, "Start Tape Cell Index: " : text px, y+45, str$(startTapeCell),,,, c

  text x+10, y+75, "Curr Program Address: "  : text px, y+75, str$(current_address),,,, c
  text x+10, y+90, "Curr Tape Cell Index: "  : text px, y+90, str$(current_tape_cell),,,, c
  text x+10, y+105, "Curr Tape Cell Value: " : text px, y+105, str$(current_tape_value),,,, c

  text x+10, y+135, "Delay (ms): "           : text px, y+135, str$(delay),,,, c

  text px, y+165, "                "
  
  if error_code = ERR_NONE then
    text x+10, y+165, "State:"               : text px, y+165, stateNames$(state),,,, c
  else
    text x+10, y+165, "State:"           : text px, y+165, error_strings$(error_code),,,, RGB(RED)
  end if
  text x+10, y+180, "Num Steps:"             : text px, y+180, str$(num_executions),,,, c

  x = CMX
  y = CMY
  for i = 1 to NCMDS
    box x, y, CMW, CMH
    select case i
      case RUNSTOP
        if state = READY or state = PAUSED then
          text x+10, y+10, "Run"
        else if state = RUNNING then
          text x+10, y+10, "Pause"
        else if state = PAUSED then
          text x+10, y+10, "Continue"
        else if state = HALTED then
          text x+10, y+10, "Completed (Reset to run again)"
        end if
      case RSTEP
        text x+10, y+10, "Step"
      case RESET
        text x+10, y+10, "Reset"
      case RMENU
        text x+10, y+10, "Return to Menu"
    end select
    y = y+CMH+5
  next i
end sub

' Highlite the currently selected Command Menu Item
sub HighliteTabularMenu x, which
  local y
  if prev_rmchoice > 0 then
    y = CMY + (prev_rmchoice-1)*(CMH+5) + CMH\2
    triangle x, y, x-8, y-5, x-8, y+5, RGB(BLACK), RGB(BLACK)
  end if
  y = CMY + (which-1)*(CMH+5) + CMH\2
  triangle x, y, x-8, y-5, x-8, y+5, RGB(RED), RGB(RED)
  prev_rmchoice = which
end sub

' Handle Command Menu selection events
sub HandleTabularMenuEvents
  local z$
  local x, pstate

  DrawTabularScreen x
  state = READY
  rmchoice = RUNSTOP
  HighliteTabularMenu x-3, rmchoice
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case DOWN
        rmchoice = rmchoice+1
        if rmchoice > NCMDS then
          rmchoice = 1
        end if
        HighliteTabularMenu x-3, rmchoice        
      case UP
        rmchoice = rmchoice-1
        if rmchoice < 1 then
          rmchoice = NCMDS
        end if
        HighliteTabularMenu x-3, rmchoice        
      case ENTER
        cmd = rmchoice
        select case cmd
          case RUNSTOP
            if state = READY then
              PrepForRun
            end if
            if state = READY or state = PAUSED then
              state = RUNNING
              z$ = INKEY$
              do
                z$ = INKEY$
                if z$ <> "" then
                  state = PAUSED
                  DrawTabularScreen x
                  exit do
                end if
                Cycle
                DrawTabularScreen x
                HighliteTabularMenu x-3, rmchoice
                pause delay
              loop until state = HALTED
            end if
            HighliteTabularMenu x-3, rmchoice
          case RSTEP
            pstate = state
            if state = READY or state = PAUSED then
              state = STEPPING
              Cycle
              DrawTabularScreen x
              pause 100
              HighliteTabularMenu x-3, rmchoice
              if state = STEPPING then
                state = pstate
              end if
              DrawTabularCommands
            end if
            HighliteTabularMenu x-3, rmchoice
        case RESET
            ReloadProgram
            PrepForRun
            DrawTabularScreen x
            HighliteTabularMenu x-3, rmchoice
        case RMENU
            exit do
      end select
    end select
  loop   
end sub

' Initialize Turing Machine counters, etc for a run
sub PrepForRun
  lowest_tape_cell_used = startTapeCell
  highest_tape_cell_used = startTapeCell
  most_recent_tape_cell = startTapeCell
  current_address = start_address
  num_executions = 0
  error_code = ERR_NONE
  state = READY
end sub

' Draw the Options Screen
sub DrawOptionsScreen
  local x, y, opx
  page write 1
  cls
  text MM.HRES\2, 20, "Options", "CT", 5
  x = OMXOFF
  y = OMYOFF
  opx = 500
  box x, y, OMWIDTH, OMHEIGHT
  text x+10, y+8, "Delay(ms): "
  text x+10, y+23, "Used to slow program in RUNNING mode"
  text x+10, y+38, "Use UP and DOWN arrow keys to change, ENTER when done"
  text opx, y+8, str$(delay),,,, RGB(CYAN)
  y = y + OMHEIGHT+5
  box x, y, OMWIDTH, OMHEIGHT
  text x+10, y+8, "Tape Display Size (S,M,L): "
  text x+10, y+23, "Type S, M, or L to change tape display size"
  text x+10, y+38, "Type ENTER when done"
  text opx, y+8, tapeSizes$(tape_size),,,, RGB(CYAN)
  y = y + OMHEIGHT+5
  box x, y, OMWIDTH, OMHEIGHT
  text x+10, y+8, "Return to Main Menu"
  page write 0
  page copy 1 to 0, B
end sub

' Highlite the currently selected Options Menu Item
sub HighliteOptionsMenu x, which
  local y
  if prev_opchoice > 0 then
    y = OMYOFF + (prev_opchoice-1)*(OMHEIGHT+5) + OMHEIGHT\2
    triangle x, y, x-8, y-5, x-8, y+5, RGB(BLACK), RGB(BLACK)
  end if
  y = OMYOFF + (which-1)*(OMHEIGHT+5) + OMHEIGHT\2
  triangle x, y, x-8, y-5, x-8, y+5, RGB(RED), RGB(RED)
  prev_opchoice = which
end sub

' Handle User Inputs for Options Screen
sub HandleOptionsEvents
  local z$, p$, tcmd$
  local cmd, pcmd

  DrawOptionsScreen
  HighliteOptionsMenu OMXOFF-3, 1
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        opchoice = opchoice-1
        if opchoice < 1 then opchoice = NOMENUS
        HighliteOptionsMenu OMXOFF-3, opchoice
      case DOWN
        opchoice = opchoice+1
        if opchoice > NOMENUS then opchoce = 1
        HighliteOptionsMenu OMXOFF-3, opchoice
      case ENTER
        if opchoice = 1 then
          do
            p$ = INKEY$
            do
              p$ = INKEY$
            loop until p$ <> ""
            pcmd = asc(p$)
            select case pcmd
              case UP
                delay = delay+10
                if delay > MAX_DELAY then delay = MAX_DELAY
              case DOWN
                delay = delay-10
                if delay < 0 then  delay = 0
              case ENTER
                exit do
            end select
            DrawOptionsScreen
            HighliteOptionsMenu OMXOFF-3, opchoice
          loop
        else if opchoice = 2 then    
          do
            p$ = INKEY$
            do
              p$ = INKEY$
            loop until p$ <> ""
            tcmd$ = UCASE$(p$)
            cmd = asc(p$)
            select case tcmd$
              case "S"
                tape_size = SMALL
              case "M"
                tape_size = MEDIUM
              case "L"
                tape_size = LARGE
            end select
            HighliteOptionsMenu OMXOFF-3, opchoice
            DrawOptionsScreen
          loop until cmd = ENTER
        else if opchoice = 3 then
          exit do
        end if
    end select
  loop
end sub

' Allow the User to Enter a Turing Program by Hand
sub HandleEnterProgramEvents
  local z$, m$, a$
  local cmd, eaddr, efield, w, n, sa, emode
  local elimit = NORMAL_PART_LENGTH

  emode = EDIT_PROGRAM
  eaddr = 0
  efield = 1
  if not program_loaded then
    ClearProgram
  end if
  DrawEnterProgramScreen eaddr, efield
  HighliteProgramEnterComponent eaddr, efield
  do
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        eaddr = eaddr-1
        if eaddr < 0 then eaddr = 0        
        efield = 1
        elimit = NORMAL_PART_LENGTH
      case DOWN, ENTER
        eaddr = eaddr+1
        if eaddr > NUM_ADDRESSES-1 then eaddr = NUM_ADDRESSES-1
        efield = 1
        elimit = NORMAL_PART_LENGTH
      case LEFT
        if emode = EDIT_PROGRAM then
          efield = efield-1
          if efield < 1 then efield = 1
        else
          current_tape_cell = current_tape_cell-1
          if current_tape_cell < 1 then current_tape_cell =1
        end if
      case RIGHT
        if emode = EDIT_PROGRAM then
          if efield < elimit then
            efield = efield+1
            if efield > NUM_INSTR_PARTS then efield = NUM_INSTR_PARTS
          end if
        else
          current_tape_cell = current_tape_cell+1
          if current_tape_cell > NUM_TAPE_CELLS then current_tape_cell = NUM_TAPE_CELLS
        end if
      case ENTER
        eaddr = eaddr+1
        if eaddr > NUM_ADDRESSES-1 then eaddr = NUM_ADDRESSES-1
        efield = 1 
     case DELETE
        program(eaddr+1, efield) = 0
      case RCLEAR, RCLEARL
        if emode = EDIT_PROGRAM then
          if DrawMessage(RCLEAR, "This will clear the program! Press C again to confirm") then
            ClearProgram
            eaddr = 0
            efield = 1
            max_address_used = 0
          end if
        else
          if DrawMessage(RCLEAR, "This will clear the tape! Press C again to confirm") then
            ClearTape
            current_tape_cell = TAPE_START
          end if
        end if
        DrawEnterProgramScreen eaddr, efield
        HighliteProgramEnterComponent eaddr, efield
      case XMENU, XMENUL
        if DrawMessage(XMENU, "This will save your work and go to Main Menu! Press M again to confirm") then
          SaveProgram
          exit do
        end if
      case RNAME, RNAMEL
        programName$ = GetString$("Enter Turing Machine Program Name: ")
      case RSTART, RSTARTL
        a$ = GetString$("Enter Turing Machine Program Start Address: ")
        sa = val(a$)
        if sa < 0 or sa > NUM_ADDRESSES-1 then
          Beep
        else
          start_address = sa
          DrawEnterProgramScreen eaddr, efield
          HighliteProgramEnterComponent eaddr, efield
        end if
      case RTAPEM, RTAPEML
        if emode = EDIT_PROGRAM then
          emode = EDIT_TAPE
          foo = DrawMessage(0, "Entering EDIT_TAPE Mode!")
          pause 500
        else
          emode = EDIT_PROGRAM
          foo = DrawMessage(0, "Entering EDIT_PROGRAM Mode!")
          pause 500
        end if      
      case XFIELD, XFIELDL
        if elimit = NORMAL_PART_LENGTH then
          elimit = NUM_SYMBOLS*NUM_MICRO_INSTR
          for i = 7 to 12
            if program(eaddr+1, i) = NOT_USED then
              program(eaddr+1, i) = 0
            end if
          next i
          program(eaddr+1, 10) = 3
          DrawEnterProgramScreen eaddr, efield
        else
          elimit = NORMAL_PART_LENGTH
          if emode = EDIT_PROGRAM then
            for i = 7 to 12  
              program(eaddr+1, i) = NOT_USED
            next i
            DrawEnterProgramScreen eaddr, efield
            HighliteProgramEnterComponent eaddr, efield
          end if
        end if
      case else
        if emode = EDIT_PROGRAM then
          if eaddr > max_address_used then
            max_address_used = eaddr
          end if
          select case efield
            case 1, 4, 7, 10
              w = val(z$)
              if w < 0 or w > NUM_SYMBOLS-2 then
                Beep
              else
                if (efield <> 10) then program(eaddr+1, efield) = w
                efield = efield+1
              end if
            case 2, 5, 8, 11
              m$ = UCASE$(z$)
              if m$ <> "L" and m$ <> "R" then
                Beep
              else
                if m$ = "L" then
                  program(eaddr+1, efield) = 0
                else
                  program(eaddr+1, efield) = 1
                end if
                if efield < elimit then
                  efield = efield+1
                end if
              end if
            case 3, 6, 9, 12
              if UCASE$(z$) = "H" then
                program(eaddr+1, efield) = HALT
              else
                n = program(eaddr+1, efield)
                if n = 0 or n = HALT then
                  n = val(z$)
                else if n > 0 and n < 10 then
                  n = n*10 + val(z$)
                end if
                if n > NUM_ADDRESSES-1 then
                  beep
                else
                  program(eaddr+1, efield) = n
                  if efield < elimit then
                    efield = efield+1
                  end if
                end if
              end if
          end select
          program_loaded = 1
        else
          w = val(z$)
          if w < 0 or w > NUM_SYMBOLS-1 then
            Beep
          else
            tape(current_tape_cell) = w
          end if
        end if
      end select
    DrawEnterProgramScreen eaddr, efield
    HighliteProgramEnterComponent eaddr, efield
  loop
end sub

' Screen for Allowing User to Enter a Turing Program
sub DrawEnterProgramScreen eaddr, efield
  local x, y
  page write 1
  cls
  DrawTape current_tape_cell
  DrawTable
  FillTable
  x = CMDSX-15
  y = TBYOFF
  text x, y, "Program and Tape Edit Commands:"
  text x, y+15, "-------------------------------"
  text x, y+30, "Use Up and Down Arrow keys to select Address"
  text x, y+45, "Use Left and Right Arrow keye to select Field"
  text x, y+60, "  W field (write tape): Legal values are 0,1,2,3"
  text x, y+75, "  M field (move head): Legal values are L, R"
  text x, y+90, "  N field (next address): Legal values 0..15, H"
  text x, y+120, "Use Delete key to clear a value"
  text x, y+120, "Additional Commands:"
  text x, y+135, "  'X' : edit the '2' and '3' symbol fields"
  text x, y+150, "  'N' : Enter the Program Name"
  text x, y+165, "  'S' : Enter the Program start address 0..15"
  text x, y+180, "  'C' : Clear (erase) the current Program or Tape"
  text x, y+195, "  'M' : Save work and return to Main Menu"
  text x, y+225, "  'T' : Enter or Exit Tape Edit Mode"
  text x, y+240, "Tape Edit Commands:"
  text x, y+255, "  Use Left and Right Arrow keys to Move Head"
  text x, y+270, "  Type 0, 1, 2, or 3 to set the value of tape cell"
  text x, y+285, "  Type 'T' again to go back to program edit mode"
  page write 0
  page copy 1 to 0, B
end sub

' Highlite the currently selected Program Enter Screen
' Selected Address and Selected Component.
sub HighliteProgramEnterComponent eaddr, efield
  local x, y, ys, xs, xf, yf, pn, xw, yh
  x = TBXOFF-3
  xs = x + ACOLW
  ys = TBYOFF + BAN1H + BAN2H
  if prev_eaddr_choice > 0 then
    y = ys + prev_epaddr_choice*ROWH + ROWH\2
    triangle x, y, x-8, y-5, x-8, y+5, RGB(BLACK), RGB(BLACK)
  end if
  y = ys + eaddr*ROWH + ROWH\2
  triangle x, y, x-8, y-5, x-8, y+5, RGB(RED), RGB(RED)
  prev_eaddr_choice = eaddr
  x = TBXOFF+ACOLW+1
  y = y - ROWH\2
  if prev_efield_choice > 0 then
    GetEFieldCoords prev_efield_choice, x, y, xf, yf, xw, yh
    box xf, yf, xw, yh,, RGB(BLACK)
  end if
  GetEFieldCoords efield, x, y, xf, yf, xw, yh
  box xf, yf, xw, yh,, RGB(YELLOW)
  prev_efield_choice = efield
end sub

' Compute the x,y coordinates and size for the highlite
' Box for Turing Program Editing
sub GetEFieldCoords efield, x, y, xf, yf, xw, yh
  local pn, mpn
  pn = (efield-1)\NUM_MICRO_INSTR
  mpn = (efield-1) mod NUM_MICRO_INSTR
  xf = x + pn*PSW + 1
  yf = y+1
  select case mpn
    case 0
      xf = xf + 0
      xw = WMCOLW-3
    case 1
      xf = xf + WMCOLW
      xw = WMCOLW-3
    case 2
      xf = xf + 2*WMCOLW
      xw = ACOLW-3
  end select
  yh = ROWH-2
end sub

' Signal a mistake
sub Beep
  play sound 1, B, T, 800, 25
  pause 200
  play stop
end sub

' Print Advisory message at screen bottom and get a confirmation
function DrawMessage(code, msg$)
  local w$
  local response
  text 10, 560, SPACE$(90)
  text 10, 560, msg$
  if code <> 0 then
    w$ = INKEY$
    do
      w$ = INKEY$
    loop until w$ <> ""
    response = asc(UCASE$(w$))
    if response = code then
      DrawMessage = 1
    else
      DrawMessage = 0
    end if
  else
    DrawMessage = 1
  end if
end function

' Function to prompt for and accept a string
function GetString$(msg$)
  text 10, 560, SPACE$(90)
  text 10, 560, msg$
  input s$
  GetString$ = s$
end function
  
' Execute one instruction cycle for the Turing Machine:
' Read current tape cell, write a new value, move the tape,
' and advance to the specified next address.
sub Cycle
  local w, m, a, p
  local word(NUM_INSTR_PARTS)

  current_tape_value = tape(current_tape_cell)
  math slice program(), current_address+1,, word()
  p = current_tape_value*NUM_MICRO_INSTR
  w = word(p+1)
  m = word(p+2)
  a = word(p+3)
  tape(current_tape_cell) = w
  most_recent_tape_cell = current_tape_cell
  if m = 1 then
    current_tape_cell = current_tape_cell+1
    if current_tape_cell > NUM_TAPE_CELLS then
      state = HALTED
      Beep
      error_code = ERR_TAPER
      exit sub
    end if
    if current_tape_cell < lowest_tape_cell_used then
      lowest_tape_cell_used = current_tape_cell
    end if
  else
    current_tape_cell = current_tape_cell-1
    if current_tape_cell < 1 then
      state = HALTED
      Beep
      error_code = ERR_TAPEL
      exit sub
    end if
    if current_tape_cell > highest_tape_cell_used then
      highest_tape_cell_used = current_tape_cell
    end if
  end if
  num_executions = num_executions+1
  if a = HALT then 
    state = HALTED
    exit sub
  else
    current_address = a
  end if
end sub

' Clear the tape
sub ClearTape
  local i
  for i = 1 to NUM_TAPE_CELLS
    tape(i) = 0
  next i
end sub

' Clear the program
sub ClearProgram
  local i, j
  for i = 1 to NUM_ADDRESSES
    for j = 1 to NUM_INSTR_PARTS
      if j <= 6 then
        program(i, j) = 0
      else
        program(i, j) = NOT_USED
      end if
    next j
  next i
  max_address_used = 0
  programName$ = ""
  program_loaded = 0
end sub

' Show various help screens
sub DrawHelpScreen
  cls
end sub

#include "Turing_Help.inc"

